%% Supplementary software for the article:
%
% "Photon count estimation in single-molecule localization microscopy"
% Rasmus . Thorsen, Christiaan N. Hulleman, Sjoerd Stallinga and Bernd Rieger
%
% This .m file reproduces the data of Figure 1A
% - (Example 1) Photon count estimation
% 1) Load experimental (aberration-corrected) PSF
% 2) Estimate photon count by vectorial PSF (MLE-)fit
% 3) Estimate photon count by Gaussian (MLE-)fit
% 4) Estimate photon count by TRABI
% 
% - figure 1
% Estimated photon count by vectorial fit, gaussian fit and TRABI as a function
% of analysis area on the camera.

% (C) Copyright 2018
% All rights reserved
% Department of Imaging Physics
% Faculty of Applied Sciences
% Delft University of Technology
% Delft, The Netherlands

%% Example 1 - initialize
clearvars
close all
clc

help example1

ROI_vec = [7 11 15 19 25 29]; % region of interest [px]
radius_vec = [4 7 11 15 19 25 29]/2; % aperture radius [px]

%% load experimental PSF (aberration corrected)
addpath('exampledata') % add data folder
load('bead45nm_8_31x31x21_teritary_spot1','allspots','parameters')
spot = allspots(:,:,round(parameters.Mz/2)+round(parameters.zemit/parameters.pixelsize)); % focused spot
Nph = parameters.signalphotoncount; % estimated photon count
Nbg = parameters.backgroundphotoncount; % estimated background photons per pixel
aberration_vector = parameters.aberrations; % estimated aberration coefficietns
clear parameters allspots
rmpath('exampledata') % remove data folder

%% Photon count estimation
Nph_vector = zeros(1,length(ROI_vec));
Nph_gaussian = zeros(1,length(ROI_vec));
Nph_trabi = zeros(1,length(radius_vec));

for nstep = 1:length(ROI_vec)
    %%%% vector fit
    addpath('matlabfun/vector fitter') % add vector scripts
    parameters = set_parameters_xy; % set parameters
    parameters.aberrations = [1,1,0.0; 1,-1,-0.0; 2,0,-0.0; aberration_vector];
    theta_vector = get_vectorfit(spot,ROI_vec(nstep),parameters); % output: theta_vector, estimated (x,y,Nph,Nbg)
    rmpath('matlabfun/vector fitter') % add vector scripts
    Nph_vector(nstep) = theta_vector(end-1);
end

for nstep = 1:length(ROI_vec)
    %%%% Gaussian fit
    addpath('matlabfun/gaussian fitter') % add gaussian fitt scripts
    theta_gaussian = get_gaussfitter(spot,ROI_vec(nstep),parameters); % output: theta_gaussian, estimated (x,y,z,sigma,Nph,Nbg)
    rmpath('matlabfun/gaussian fitter') % remove gaussian fit scripts
    Nph_gaussian(nstep) = theta_gaussian(end-1);
end

for nstep = 1:length(radius_vec)
    %%%% TRABI
    addpath('matlabfun/trabi') % add trabi scripts
    % radius = 214.5*1.86/parameters.pixelsize; % aperture radius in trabi [px]
    Nph_trabi(nstep) = get_trabi(spot,radius_vec(nstep),Nbg,parameters); % output: theta_trabi, estimated (Nph)
    rmpath('matlabfun/trabi') % remove trabi scripts
end

%% Plot
%%% Photon counts (vectorial, gaussian, trabi)
scrsz = [1 1 1366 768];
figure
set(gcf,'Position',[.5*scrsz(4) 0.25*scrsz(4) 1.0*scrsz(4) 0.3*scrsz(4)]);

subplot(1,3,1)
plot(ROI_vec,Nph_vector)
ylabel('Estimated photon count')
xlabel('Region of interest [px]')
title('Vector PSF fit')
axis square
ylim([5.5 9.5]*1e3)
xlim([5 30])

subplot(1,3,2)
plot(ROI_vec,Nph_gaussian)
ylabel('Estimated photon count')
xlabel('Region of interest [px]')
title('Gaussian fit')
axis square
ylim([5.5 9.5]*1e3)
xlim([5 30])

subplot(1,3,3)
plot(radius_vec*2,Nph_trabi)
ylabel('Estimated photon count')
xlabel('Aperture diameter [px]')
title('TRABI')
axis square
ylim([5.5 9.5]*1e3)
xlim([5 30])
